// ==========================================================================
//
// = LIBRARY
//     PCIECONFIGSPACE
//
// = FILENAME
//     pciesimgmt.cpp
//
// = COPYRIGHT
//     (C) Copyright 2003 Agilent Technologies
//
// ==========================================================================
#include "AgtPCIPort.h"
#include "devpciedefs.h"
#include "devpcieintdefs.h"
#include "PCIESiMgmt.h"
#include "PCIEFPGAFramework.h"
#include "PCIEFPGACSPkt.h"

#define PCIE_SIMGMT_TIMOUT_FOR_SENDING_PACKETS ( ( UInt32 ) 100 )

//---------------------------------------------------------------------------
CPCIESiMgmt::CPCIESiMgmt()
{
  // SiDefaultSet(); // Already done on start-up
  for( UInt32 i = 0; i < PCIE_MAXIMUM_NUMBER_OF_PROBE_BOARDS_COPY3; i++ )
  {
    mSuccessfullySent[i] = false;
  }
}

//---------------------------------------------------------------------------
CPCIESiMgmt::~CPCIESiMgmt()
{
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiReqPropWrite
(
  const AgtPortHandleT& portHandle,
  const EPCIESi& prop, 
  const UInt32& val
)
{
  SiPropWrite( portHandle, 1, prop, val );
}

/*----------------------------------------------------------------*/
UInt32 CPCIESiMgmt::SiReqPropRead
(
  const AgtPortHandleT& portHandle,
  const EPCIESi& prop
)
{
  return SiPropRead( portHandle, 1, prop );
}

/*----------------------------------------------------------------*/
UInt32 CPCIESiMgmt::SiCompPropRead
(
 const AgtPortHandleT& portHandle, 
 const EPCIESi& prop
)
{
  return SiPropRead(portHandle,0,prop);
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiCompPropWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIESi& prop, 
 const UInt32& val
)
{
  SiPropWrite(portHandle,0,prop,val);
}


/*----------------------------------------------------------------*/
UInt32 CPCIESiMgmt::SiCompPyldGet(const AgtPortHandleT& portHandle, const UInt8& offset)
{
  return SiPyldGet(portHandle,0,offset);
}

/*----------------------------------------------------------------*/
UInt32 CPCIESiMgmt::SiReqPyldGet(const AgtPortHandleT& portHandle, const UInt8& offset)
{
  return SiPyldGet(portHandle,1,offset);
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiReqPyldSet(const AgtPortHandleT& portHandle, const UInt8& offset,const UInt32& val)
{
  SiPyldSet(portHandle,1,offset,val);
}


/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiPropWrite
(
 const AgtPortHandleT& portHandle, 
 const bool& req,
 const EPCIESi& prop, 
 const UInt32& val
 )
{
  if(req)
  {
    switch(prop)
    {
    case PCIE_SI_AUTOTAG:
      CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiCtrl( portHandle,
        CPCIEFpgaCSPkt::SICTRL_SWTAG, val ? 0 : 1 );
      break;
    case PCIE_SI_RESOURCE:
      CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiCtrl( portHandle,
        CPCIEFpgaCSPkt::SICTRL_RESOURCE, val );
      break;
    case PCIE_SI_INTADDR:
      CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiIntAddr( portHandle, val );
      break;
    case PCIE_SI_BUSERROR:
      CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiCtrl( portHandle,
        CPCIEFpgaCSPkt::SICTRL_BUSERROR, val );
      break;
    default:
      CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiHeaderProp( portHandle, prop, val );
    }
  }
}

/*----------------------------------------------------------------*/
UInt32 CPCIESiMgmt::SiPropRead
(
 const AgtPortHandleT& portHandle, 
 const bool& req, 
 const EPCIESi& prop
 )
{
  if(req)
  {
    switch(prop)
    {
    case PCIE_SI_AUTOTAG:
      return ( CPCIEFpgaFramework::instance().getFpgaCSPkt().getSiCtrl( portHandle,
        CPCIEFpgaCSPkt::SICTRL_SWTAG ) ? 0 : 1 );
    case PCIE_SI_RESOURCE:
      return CPCIEFpgaFramework::instance().getFpgaCSPkt().getSiCtrl( portHandle,
        CPCIEFpgaCSPkt::SICTRL_RESOURCE );
    case PCIE_SI_INTADDR:
      return CPCIEFpgaFramework::instance().getFpgaCSPkt().getSiIntAddr( portHandle );
    case PCIE_SI_BUSERROR:
      return CPCIEFpgaFramework::instance().getFpgaCSPkt().getSiCtrl( portHandle,
        CPCIEFpgaCSPkt::SICTRL_BUSERROR );
    case PCIE_SI_FUNCTION_SELECTED:
      return CPCIEFpgaFramework::instance().getFpgaCSPkt().getValueSendImmediateArray( portHandle );
    default:
      return CPCIEFpgaFramework::instance().getFpgaCSPkt().getSiHeaderProp( portHandle, prop );
    }
  }
  else
  {
    switch( prop )
    {
    case PCIE_SI_AUTOTAG:
    case PCIE_SI_RESOURCE:
    case PCIE_SI_INTADDR:
    case PCIE_SI_BUSERROR:
      return 0x0;
    default:
      return CPCIEFpgaFramework::instance().getFpgaCSSpc().getSiComplHeaderProp( portHandle, prop );
    }
  }
  
  return 0x0;
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiSend(const AgtPortHandleT& portHandle)
{
  UInt32 timeout = 0;
  bool valid = 1;

  UInt32 type = SiReqPropRead(portHandle, PCIE_SI_TYPE);
 
  /* clear si completion packet available status */
  CPCIEFpgaFramework::instance().getFpgaCSSpc().setSiComplStatus(portHandle,1);
  
  if( (type == 0) || (type == 1) || (type == 2) ) // is Memory or IO access
  { 
    // Check bus master enable bit
    UInt32 tmp = CPCIEFpgaFramework::instance().getConfigSpace().ConfigSpaceDirectRead( portHandle,
      CPCIEConfigSpace::CONFIGSPACEENTRYOFFSET_DW_STATUS_COMMAND * 4 );
    if( ((tmp >> 2) & 0x1) == 0)
    {
      valid = 0;
      mSuccessfullySent[portHandle] = false;
    }    
  }

  if( valid )
  {
    /* initiate a send immediate */
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiCtrl(portHandle,CPCIEFpgaCSPkt::SICTRL_SEND,1);

    UInt32 sendStatus = 1;

    while( (timeout < PCIE_SIMGMT_TIMOUT_FOR_SENDING_PACKETS) && sendStatus == 1)
    {
      sendStatus = CPCIEFpgaFramework::instance().getFpgaCSPkt().getSiCtrl(portHandle,CPCIEFpgaCSPkt::SICTRL_SEND);
      timeout++;
      //Sleep(10);
    }
    
    if( sendStatus == 1)
    {
      mSuccessfullySent[portHandle] = false;
    }
    else
    {
      mSuccessfullySent[portHandle] = true;
    }
  }

}


/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiDefaultSet(const AgtPortHandleT& portHandle)
{
  // first dword
  SiPropWrite(portHandle,1,PCIE_SI_LEN,0x1);
  SiPropWrite(portHandle,1,PCIE_SI_RSVD023,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_ATTR,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_EP,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_TD,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_RSVD013,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_TC,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_RSVD017,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_TYPE,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_FMT,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_RSVD007,0x0);

  // second dword
  SiPropWrite(portHandle,1,PCIE_SI_1STDWBE,0xf);
  SiPropWrite(portHandle,1,PCIE_SI_LASTDWBE,0xf);
  SiPropWrite(portHandle,1,PCIE_SI_AUTOTAG,0x1);
  SiPropWrite(portHandle,1,PCIE_SI_TAG,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_REQID,0x0);

 
  // third dword
  SiPropWrite(portHandle,1,PCIE_SI_MEM64_ADDRHI,0x0);

  // fourth dword
  SiPropWrite(portHandle,1,PCIE_SI_MEM64_ADDRLO,0x0);
  SiPropWrite(portHandle,1,PCIE_SI_MEM64_RSVD151,0x0);
  
  // behaviour
  SiPropWrite(portHandle,1,PCIE_SI_RESOURCE,(UInt32)PCIE_SIV_RESOURCE_SIMEM);
  SiPropWrite(portHandle,1,PCIE_SI_BUSERROR,0x0);


  CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiDataDW(portHandle,0x0,0x0);
  CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiDataDW(portHandle,0x1,0x0);
  CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiDataDW(portHandle,0x2,0x0);
  CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiDataDW(portHandle,0x3,0x0);

}

/*----------------------------------------------------------------*/
UInt32 CPCIESiMgmt::SiPyldGet(const AgtPortHandleT& portHandle, const bool& req,const UInt8& offset)
{
  if(req)
  {
    return CPCIEFpgaFramework::instance().getFpgaCSPkt().getSiDataDW(portHandle,offset);
  }
  else
  {
    return CPCIEFpgaFramework::instance().getFpgaCSSpc().getSiComplPayloadDW(portHandle,offset);
  }
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiPyldSet(const AgtPortHandleT& portHandle,const bool& req,const UInt8& offset,const UInt32& val)
{
  if(req)
  {
     CPCIEFpgaFramework::instance().getFpgaCSPkt().setSiDataDW(portHandle,offset,val);
  }
  else
  {
    //CPCIEFpgaFramework::instance().getFpgaRxMisc().setSiComplPayloadDW(portHandle,offset,val);
  }
}

/*----------------------------------------------------------------*/
UInt32 CPCIESiMgmt::SiStatusGet(const AgtPortHandleT& portHandle, const EPCIESiStatus& siStatus)
{
  switch(siStatus)
  {
  case PCIE_SISTATUS_COMP_AVAILABLE:
    //return mCompletionAvailable;
    AGT_THROW("CPCIESiMgmt::SiStatusGet not implemented for FPGA");
  case PCIE_SISTATUS_SENT:
    return (mSuccessfullySent[portHandle] ? 0x1 : 0x0);    
  default:
    return 0x0;
  }
  return 0x0;
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiCompPyldByteGet(const AgtPortHandleT& portHandle,  UInt32 numBytes, UInt8* theData)
{
  SiPyldByteGet(portHandle,false,numBytes,theData);
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiReqPyldByteGet(const AgtPortHandleT& portHandle,  UInt32 numBytes, UInt8* theData)
{
  SiPyldByteGet(portHandle,true,numBytes,theData);
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiPyldByteGet(const AgtPortHandleT& portHandle, const bool& req, UInt32 numBytes, UInt8* theData)
{
  UInt32 dw[4] = {0,0,0,0};
  for (UInt8 i = 0; i < (numBytes/4);i++)
  {
    if(req)
    {
      dw[i] = SiReqPyldGet(portHandle,i);
    }
    else
    {
      dw[i] = SiCompPyldGet(portHandle,i);
    }
  }    
  for (UInt8 i = 0; i < numBytes; i++)
  {
    theData[i] = UInt8 (( dw[(i/4)] >> (8*(i % 4)) ) & 0xff);
  }
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiReqPyldByteSet(const AgtPortHandleT& portHandle,  UInt32 numBytes, UInt8* theData)
{
  SiPyldByteSet(portHandle,true,numBytes,theData);
}

/*----------------------------------------------------------------*/
void CPCIESiMgmt::SiPyldByteSet(const AgtPortHandleT& portHandle, const bool& req, UInt32 numBytes, UInt8* theData)
{
  UInt32 dw[4] = {0,0,0,0};
  for (UInt8 i = 0; i < numBytes; i++)
  {
    dw[(i/4)] |= theData[i] << ( (i%4) *8);
  }

  for (UInt8 i = 0; i < (numBytes/4);i++)
  {
    if(req)
    {
      SiReqPyldSet(portHandle,i,dw[i]);
    }
    else
    {
      // not possible
    }
  }
}
